/* global Vue */

(function () {
  function qsToObj(search) {
    const params = new URLSearchParams(search || window.location.search);
    const obj = {};
    for (const [k, v] of params.entries()) {
      if (k.endsWith('[]')) {
        const key = k.slice(0, -2);
        obj[key] = obj[key] || [];
        obj[key].push(v);
      } else {
        obj[k] = v;
      }
    }
    return obj;
  }

  function normalizeSearchParams(raw) {
    const p = { ...raw };

    // Dates
    if (p.start && !p.start_date) p.start_date = p.start;
    if (p.end && !p.end_date) p.end_date = p.end;

    // Guests
    p.adults = parseInt(p.adults || '2', 10);
    if (!Number.isFinite(p.adults) || p.adults < 1) p.adults = 2;

    p.children = parseInt(p.children || '0', 10);
    if (!Number.isFinite(p.children) || p.children < 0) p.children = 0;

    // child_ages[]
    let ages = p.child_ages || p.children_ages || [];
    if (typeof ages === 'string') {
      ages = ages.split(',').filter(Boolean);
    }
    if (!Array.isArray(ages)) ages = [];
    ages = ages.map((x) => parseInt(x, 10)).filter((x) => Number.isFinite(x));
    // fill/trim to children count
    while (ages.length < p.children) ages.push(5);
    if (ages.length > p.children) ages = ages.slice(0, p.children);

    p.child_ages = ages;

    return p;
  }

  function toQuery(params) {
    const q = new URLSearchParams();
    Object.entries(params).forEach(([k, v]) => {
      if (v === undefined || v === null || v === '') return;
      if (Array.isArray(v)) {
        v.forEach((item) => q.append(k + '[]', String(item)));
      } else {
        q.set(k, String(v));
      }
    });
    return q.toString();
  }

  async function apiGet(url, params) {
    const q = toQuery(params || {});
    const full = q ? `${url}?${q}` : url;
    const res = await fetch(full, {
      headers: {
        'Accept': 'application/json',
      },
      credentials: 'same-origin',
    });

    const json = await res.json().catch(() => null);
    if (!res.ok) {
      const msg = json?.message || `HTTP ${res.status}`;
      throw new Error(msg);
    }
    return json;
  }

  const mount = document.getElementById('booking-hotels-spa-search');
  if (!mount || typeof Vue === 'undefined') return;

  const initial = normalizeSearchParams({
    ...qsToObj(window.location.search),
    ...(mount.dataset.initial ? JSON.parse(mount.dataset.initial) : {}),
  });

  Vue.createApp({
    data() {
      return {
        loading: false,
        error: null,
        search: initial,
        hotels: [],
      };
    },
    computed: {
      hasSearch() {
        return !!(this.search.start_date && this.search.end_date);
      },
    },
    methods: {
      async load() {
        this.loading = true;
        this.error = null;
        try {
          const res = await apiGet('/api/booking-hotels/search', this.search);
          this.hotels = res?.data?.hotels || [];
        } catch (e) {
          this.error = e?.message || 'Failed to load';
          this.hotels = [];
        } finally {
          this.loading = false;
        }
      },
      detailUrl(item) {
        const slug = item?.hotel?.slug;
        const token = item?.offer_token;
        const q = {
          start_date: this.search.start_date,
          end_date: this.search.end_date,
          adults: this.search.adults,
          children: this.search.children,
        };
        if (Array.isArray(this.search.child_ages) && this.search.child_ages.length) {
          q.child_ages = this.search.child_ages.join(',');
        }
        if (token) q.offer_token = token;
        return `/booking-hotels/${encodeURIComponent(slug)}?${toQuery(q)}`;
      },
      money(x) {
        const n = Number(x || 0);
        return n.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: 2 });
      },
    },
    mounted() {
      if (this.hasSearch) {
        this.load();
      }
    },
    template: `
      <div class="container" style="padding: 30px 0;">
        <div v-if="error" class="alert alert-danger">{{ error }}</div>
        <div v-if="!hasSearch" class="alert alert-info">Please select dates and guests to search.</div>

        <div v-if="loading" class="alert alert-secondary">Loading...</div>

        <div v-if="!loading && hasSearch">
          <div class="d-flex justify-content-between align-items-center mb-3">
            <h3 class="m-0">Results ({{ hotels.length }})</h3>
          </div>

          <div v-if="hotels.length === 0" class="alert alert-warning">No hotels found.</div>

          <div v-for="item in hotels" :key="item.hotel.id" class="card mb-3">
            <div class="card-body">
              <div class="d-flex justify-content-between align-items-start">
                <div>
                  <h5 class="card-title mb-1">{{ item.hotel.title }}</h5>
                  <div class="text-muted" style="font-size: 13px;">
                    {{ item.hotel.location?.name || '' }}
                  </div>
                </div>
                <div class="text-end">
                  <div class="text-muted" style="font-size: 12px;">Total</div>
                  <div style="font-size: 20px; font-weight: 700;">{{ money(item.best_offer.pricing.total) }} {{ item.best_offer.pricing.currency }}</div>
                  <div class="text-muted" style="font-size: 12px;">{{ item.best_offer.pricing.nights }} nights</div>
                </div>
              </div>

              <div class="mt-3 d-flex justify-content-end">
                <a class="btn btn-primary" :href="detailUrl(item)">View offers</a>
              </div>
            </div>
          </div>
        </div>
      </div>
    `,
  }).mount('#booking-hotels-spa-search');
})();
