/**
 * Room Selection Manager
 * Manages room selection with rate plans and quantities
 */
class RoomSelectionManager {
    constructor() {
        this.selectedRooms = []; // [{roomId, ratePlanId, quantity, price, planName, ...}]
        this.searchParams = window.bookingSearchParams || {};
        this.init();
    }

    init() {
        // Event listeners
        // ✅ FIX: Support both .room-quantity-select (new) and .room-quantity (old) for backward compatibility
        $(document).on('change', '.room-quantity-select, .room-quantity', this.handleQuantityChange.bind(this));
        $(document).on('click', '.select-plan-btn', this.handlePlanSelection.bind(this));
        $(document).on('click', '.book-now-btn', this.handleBookNow.bind(this)); // ✅ NEW: Book Now button
        $(document).on('click', '.remove-room', this.handleRoomRemoval.bind(this)); // ✅ Fix: use .remove-room instead of .remove-room-btn
        $(document).on('click', '#clear-selection-btn', this.handleClearSelection.bind(this));
        $(document).on('click', '#proceed-to-checkout-btn', this.handleProceedToCheckout.bind(this));
    }

    /**
     * Handle quantity change for a room
     * ✅ FIX: Only updates quantity for the specific room, not other rooms
     */
    handleQuantityChange(e) {
        const $select = $(e.target);
        const roomId = parseInt($select.data('room-id'));
        const quantity = parseInt($select.val()) || 1;

        // ✅ FIX: Validate roomId
        if (!roomId || isNaN(roomId)) {
            console.error('Invalid roomId in quantity change handler', $select);
            return;
        }

        // ✅ FIX: Only update price for rate plans in THIS specific room
        // Use more specific selector to ensure we only get plans for this room
        const $roomCard = $(`.room-with-plans-card[data-room-id="${roomId}"]`);
        if (!$roomCard.length) {
            console.warn('Room card not found for room', roomId);
            return;
        }

        // ✅ FIX: Support both rate-plan-item and rate-plan-card-improved classes
        $roomCard.find(`[data-room-id="${roomId}"][data-plan-id]`).each((index, element) => {
            let $planCard = $(element).closest('.rate-plan-item, .rate-plan-card-improved');
            if (!$planCard.length) {
                $planCard = $(element);
            }
            const planRoomId = parseInt($planCard.data('room-id'));

            // ✅ Double check: ensure we're only updating plans for this room
            if (planRoomId !== roomId) {
                return; // Skip if room ID doesn't match
            }

            const basePrice = parseFloat($planCard.data('base-price'));
            if (isNaN(basePrice) || basePrice <= 0) {
                return; // Skip if invalid base price
            }

            const newTotal = basePrice * quantity;

            // Update displayed price
            $planCard.find('.total-price').text(this.formatMoney(newTotal));

            // Update selected room if this plan is selected
            const planId = parseInt($planCard.data('plan-id'));
            const selectedRoom = this.selectedRooms.find(r =>
                parseInt(r.roomId) === roomId && parseInt(r.ratePlanId) === planId
            );

            if (selectedRoom) {
                selectedRoom.quantity = quantity;
                selectedRoom.totalPrice = newTotal;
                selectedRoom.pricePerNight = selectedRoom.basePrice / (this.searchParams.nights || 1);
                this.updateSummary();
            }
        });
    }

    /**
     * ✅ NEW: Log to separate test log file
     */
    logToTestFile(level, message, data = {}) {
        const timestamp = new Date().toISOString();
        const logData = {
            timestamp: timestamp,
            level: level,
            page: 'ROOM_SELECTION',
            message: message,
            data: data,
            url: window.location.href,
            searchParams: this.searchParams
        };

        // Log to console with special marker
        console.log(`[TEST LOG - ${level}] ${message}`, logData);

        // Send to server for logging (if endpoint exists)
        if (window.bookingTestLogEndpoint) {
            fetch(window.bookingTestLogEndpoint, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
                },
                body: JSON.stringify(logData)
            }).catch(err => console.warn('Failed to send test log to server:', err));
        }
    }

    /**
     * Handle rate plan selection
     */
    handlePlanSelection(e) {
        e.preventDefault();
        const $btn = $(e.currentTarget);
        const roomId = parseInt($btn.data('room-id'));
        const planId = parseInt($btn.data('plan-id'));
        const planName = $btn.data('plan-name');
        const basePrice = parseFloat($btn.data('base-price'));
        const pricePerNight = parseFloat($btn.data('price-per-night'));
        const offerToken = $btn.data('offer-token');
        const mealOption = $btn.data('meal-option');
        const isRefundable = $btn.data('is-refundable') == 1;

        // ✅ TEST LOG: Plan selection started
        this.logToTestFile('INFO', 'Rate plan selection started', {
            roomId: roomId,
            planId: planId,
            planName: planName,
            basePrice: basePrice,
            pricePerNight: pricePerNight,
            offerToken: offerToken,
            mealOption: mealOption,
            isRefundable: isRefundable
        });

        // ✅ FIX: Get quantity from the specific room's quantity selector only
        // Support both .room-quantity-select (new) and .room-quantity (old) for backward compatibility
        const $quantitySelect = $(`.room-quantity-select[data-room-id="${roomId}"], .room-quantity[data-room-id="${roomId}"]`).first();
        if (!$quantitySelect.length) {
            console.error('Quantity selector not found for room', roomId);
            return;
        }
        const quantity = parseInt($quantitySelect.val()) || 1;

        // ✅ FIX: Validate quantity
        if (quantity <= 0 || isNaN(quantity)) {
            console.error('Invalid quantity', quantity);
            return;
        }

        // ✅ FIX: Use more specific selector - find room card first, then plan within it
        const $roomCard = $(`.room-with-plans-card[data-room-id="${roomId}"]`);
        if (!$roomCard.length) {
            console.error('Room card not found', { roomId });
            return;
        }

        // ✅ FIX: Support both rate-plan-item and rate-plan-card-improved classes
        // First try to find the element with data attributes
        let $planCard = $roomCard.find(`[data-plan-id="${planId}"][data-room-id="${roomId}"]`);

        // If found, get the closest card element
        if ($planCard.length) {
            $planCard = $planCard.closest('.rate-plan-item, .rate-plan-card-improved, .card');
        }

        // If still not found, try direct selector
        if (!$planCard.length) {
            $planCard = $roomCard.find(`.rate-plan-item[data-plan-id="${planId}"], .rate-plan-card-improved[data-plan-id="${planId}"]`);
        }

        if (!$planCard.length) {
            console.error('Plan card not found within room card', { planId, roomId, roomCardExists: $roomCard.length });
            return;
        }

        // ✅ FIX: Double check that we have the correct room ID
        const cardRoomId = parseInt($planCard.data('room-id'));
        const cardPlanId = parseInt($planCard.data('plan-id'));
        if (cardRoomId !== roomId || cardPlanId !== planId) {
            console.error('Room ID or Plan ID mismatch', {
                expected: { roomId, planId },
                actual: { roomId: cardRoomId, planId: cardPlanId }
            });
            return;
        }

        // ✅ TEST LOG: Quantity retrieved
        this.logToTestFile('INFO', 'Quantity retrieved for plan selection', {
            roomId: roomId,
            planId: planId,
            quantity: quantity
        });

        // Process the selection
        this.processPlanSelection($planCard, roomId, planId, planName, basePrice, pricePerNight, offerToken, mealOption, isRefundable, quantity);
    }

    /**
     * ✅ NEW: Process plan selection (extracted for reusability)
     */
    processPlanSelection($planCard, roomId, planId, planName, basePrice, pricePerNight, offerToken, mealOption, isRefundable, quantity) {
        const totalPrice = basePrice * quantity;

        // Remove any previous selection for this room
        this.selectedRooms = this.selectedRooms.filter(r => r.roomId != roomId);

        // ✅ FIX: Get room name - try multiple selectors (updated for new design system)
        let roomName = '';
        const $roomCard = $(`.room-with-plans-card[data-room-id="${roomId}"]`);
        if ($roomCard.length) {
            // Try new design system first (.hotel-title)
            roomName = $roomCard.find('.hotel-title').text().trim();
            // Fallback to old layouts
            if (!roomName) {
                roomName = $roomCard.find('.card-header-highlighted h5, .room-name').text().trim();
            }
        }
        if (!roomName) {
            roomName = 'Room #' + roomId;
        }

        // Add new selection
        const roomData = {
            roomId: roomId,
            ratePlanId: planId,
            quantity: quantity,
            totalPrice: totalPrice,
            basePrice: basePrice,
            pricePerNight: pricePerNight,
            planName: planName,
            offerToken: offerToken || null,
            mealOption: mealOption ? (typeof mealOption === 'string' ? JSON.parse(mealOption) : mealOption) : null,
            isRefundable: isRefundable,
            roomName: roomName,
        };

        console.log('✅ Adding room to selection:', roomData); // ✅ Debug log

        // ✅ TEST LOG: Room added to selection
        this.logToTestFile('INFO', 'Room added to selection', {
            roomData: roomData,
            calculation: {
                basePrice: basePrice,
                quantity: quantity,
                totalPrice: totalPrice,
                formula: `${basePrice} × ${quantity} = ${totalPrice}`
            },
            selectedRoomsCount: this.selectedRooms.length + 1
        });

        this.selectedRooms.push(roomData);

        // Update UI
        this.updatePlanSelection($planCard, true);
        this.updateSummary();
    }

    /**
     * Update plan selection visual state
     * ✅ FIXED: Properly highlight selected rate plan card and button
     */
    updatePlanSelection($planCard, isSelected) {
        const roomId = $planCard.data('room-id');
        // ✅ FIX: Use more specific selector to find all plans for this room (support both classes)
        const $roomCard = $(`.room-with-plans-card[data-room-id="${roomId}"]`);
        let $allPlans = $roomCard.find('[data-room-id="' + roomId + '"][data-plan-id]').closest('.rate-plan-item, .rate-plan-card-improved, .card');
        if (!$allPlans.length) {
            // Fallback to old selector
            $allPlans = $(`.rate-plan-item[data-room-id="${roomId}"], .rate-plan-card-improved[data-room-id="${roomId}"]`);
        }

        // Remove selection from all plans in this room - CSS rules will handle button visibility
        $allPlans.removeClass('selected');
        $allPlans.find('.select-plan-btn')
            .removeClass('btn-success')
            .addClass('btn-primary')
            .html('<i class="fa fa-check"></i> ' + this.translate('Select'))
            .css('display', ''); // Remove inline styles, let CSS handle it

        // Remove inline styles from Book Now buttons - CSS rule will handle it when .selected is removed
        $allPlans.find('.book-now-btn').css('display', '');

        // Mark selected plan
        if (isSelected && $planCard.length) {
            // Add 'selected' class to the rate-plan-item card
            $planCard.addClass('selected');
            console.log('✅ Plan card selected', {
                roomId: roomId,
                planId: $planCard.data('plan-id'),
                hasSelectedClass: $planCard.hasClass('selected')
            });

            // Remove inline styles - CSS rules will handle button visibility
            const $selectBtn = $planCard.find('.select-plan-btn');
            $selectBtn.css('display', ''); // Remove inline styles, CSS rule .rate-plan-item.selected .select-plan-btn will hide it

            // Remove inline styles from Book Now button - CSS rule .rate-plan-item.selected .book-now-btn will show it
            const $bookNowBtn = $planCard.find('.book-now-btn');
            if ($bookNowBtn.length) {
                $bookNowBtn.css('display', ''); // Remove inline styles, let CSS handle it
                console.log('✅ Buttons should be toggled via CSS rules', {
                    hasSelectedClass: $planCard.hasClass('selected'),
                    selectBtnFound: $selectBtn.length,
                    bookNowBtnFound: $bookNowBtn.length,
                    computedDisplaySelect: window.getComputedStyle($selectBtn[0]).display,
                    computedDisplayBookNow: window.getComputedStyle($bookNowBtn[0]).display
                });
            } else {
                console.warn('⚠️ Book Now button not found in plan card', {
                    planCardClasses: $planCard.attr('class'),
                    planCardHtml: $planCard.html().substring(0, 200)
                });
            }
        }
    }

    /**
     * Handle room removal from selection
     */
    handleRoomRemoval(e) {
        e.preventDefault();
        const roomId = parseInt($(e.target).closest('.selected-room-item').data('room-id'));
        this.removeRoom(roomId);
    }

    /**
     * Remove room from selection
     */
    removeRoom(roomId) {
        this.selectedRooms = this.selectedRooms.filter(r => r.roomId != roomId);

        // Update UI - support both classes
        const $roomCard = $(`.room-with-plans-card[data-room-id="${roomId}"]`);
        const $planCards = $roomCard.find(`[data-room-id="${roomId}"][data-plan-id]`).closest('.rate-plan-item, .rate-plan-card-improved');
        $planCards.removeClass('selected');
        $planCards.find('.select-plan-btn')
            .removeClass('btn-success')
            .addClass('btn-primary')
            .html('<i class="fa fa-check"></i> ' + this.translate('Select'))
            .css('display', ''); // Remove inline styles, let CSS handle it

        // Remove inline styles from Book Now buttons - CSS rule will handle it when .selected is removed
        $planCards.find('.book-now-btn').css('display', '');

        this.updateSummary();
    }

    /**
     * Handle Book Now button click
     * Directly proceed to checkout for this specific room and rate plan
     */
    handleBookNow(e) {
        e.preventDefault();
        const $btn = $(e.currentTarget);
        const roomId = parseInt($btn.data('room-id'));
        const planId = parseInt($btn.data('plan-id'));

        // Find the selected room data
        const selectedRoom = this.selectedRooms.find(r =>
            r.roomId == roomId && r.ratePlanId == planId
        );

        if (!selectedRoom) {
            // If not in selected rooms, select it first
            const $selectBtn = $(`.select-plan-btn[data-room-id="${roomId}"][data-plan-id="${planId}"]`);
            if ($selectBtn.length) {
                $selectBtn.trigger('click');
                // Wait a bit then proceed
                setTimeout(() => {
                    this.proceedToCheckoutForRoom(roomId, planId);
                }, 100);
            }
            return;
        }

        // Proceed to checkout
        this.proceedToCheckoutForRoom(roomId, planId);
    }

    /**
     * Proceed to checkout for a specific room
     */
    proceedToCheckoutForRoom(roomId, planId) {
        // Build checkout URL with this specific room
        const checkoutUrl = this.buildCheckoutUrlForRoom(roomId, planId);
        if (checkoutUrl) {
            console.log('Navigating to checkout (Book Now):', checkoutUrl); // ✅ Debug log
            window.location.href = checkoutUrl;
        } else {
            console.error('Failed to build checkout URL for room');
            alert(this.translate('Failed to build checkout URL. Please try again.'));
        }
    }

    /**
     * Build checkout URL for a specific room
     */
    buildCheckoutUrlForRoom(roomId, planId) {
        const baseUrl = window.bookingCheckoutUrl || '';
        if (!baseUrl) {
            console.error('Checkout URL not defined. window.bookingCheckoutUrl is:', window.bookingCheckoutUrl);
            this.logToTestFile('ERROR', 'Checkout URL not defined', { baseUrl: baseUrl });
            return null;
        }

        console.log('Building checkout URL for room with baseUrl:', baseUrl); // ✅ Debug log
        console.log('Current searchParams:', this.searchParams); // ✅ Debug log

        const selectedRoom = this.selectedRooms.find(r =>
            r.roomId == roomId && r.ratePlanId == planId
        );

        if (!selectedRoom) {
            console.error('Room not selected');
            this.logToTestFile('ERROR', 'Room not found in selected rooms', { roomId, planId, selectedRooms: this.selectedRooms });
            return null;
        }

        const params = new URLSearchParams();

        // ✅ FIX: Add all search params with proper handling
        this.addSearchParamsToUrl(params);

        // Add this room
        params.append('rooms[0][room_id]', selectedRoom.roomId);
        params.append('rooms[0][rate_plan_id]', selectedRoom.ratePlanId);
        params.append('rooms[0][quantity]', selectedRoom.quantity || 1);
        if (selectedRoom.offerToken) {
            params.append('rooms[0][offer_token]', selectedRoom.offerToken);
        }

        const finalUrl = baseUrl + '?' + params.toString();
        console.log('✅ Final checkout URL (Book Now):', finalUrl); // ✅ Debug log

        // ✅ TEST LOG: Checkout URL built
        this.logToTestFile('INFO', 'Checkout URL built (Book Now)', {
            roomId: roomId,
            planId: planId,
            selectedRoom: selectedRoom,
            searchParams: this.searchParams,
            finalUrl: finalUrl,
            urlParams: Object.fromEntries(params)
        });

        return finalUrl;
    }

    /**
     * ✅ NEW: Add search parameters to URL params
     * Centralized method to handle all search params consistently
     */
    addSearchParamsToUrl(params) {
        // Get search params - try multiple sources
        const searchParams = this.getCurrentSearchParams();

        console.log('✅ Adding search params to URL:', searchParams); // ✅ Debug log

        // Dates
        if (searchParams.start_date) {
            params.append('start_date', searchParams.start_date);
        }
        if (searchParams.end_date) {
            params.append('end_date', searchParams.end_date);
        }

        // Guests
        if (searchParams.adults) {
            params.append('adults', searchParams.adults);
        } else if (searchParams.adults === 0) {
            params.append('adults', '1'); // Default to 1 if not set
        }

        if (searchParams.children !== undefined && searchParams.children !== null) {
            params.append('children', searchParams.children);
        }

        // Child ages - support both child_ages and children_ages
        let childAges = searchParams.child_ages || searchParams.children_ages || [];
        if (childAges) {
            if (typeof childAges === 'string') {
                // Handle comma-separated string
                childAges = childAges.split(',').map(age => age.trim()).filter(age => age);
            }
            if (Array.isArray(childAges)) {
                childAges.forEach(age => {
                    if (age !== '' && age !== null && age !== undefined) {
                        params.append('child_ages[]', age);
                        params.append('children_ages[]', age); // ✅ Also add children_ages for compatibility
                    }
                });
            }
        }

        console.log('✅ Search params added to URL'); // ✅ Debug log
    }

    /**
     * ✅ NEW: Get current search params from multiple sources
     */
    getCurrentSearchParams() {
        // Try to get from instance variable first (if already loaded)
        let params = this.searchParams || window.bookingSearchParams || {};

        // If empty, try to get from window (initial load)
        if (!params || Object.keys(params).length === 0) {
            params = window.bookingSearchParams || {};
        }

        // Try to get from URL params (if page was refreshed or navigated)
        const urlParams = new URLSearchParams(window.location.search);
        if (urlParams.has('start_date') && !params.start_date) {
            params.start_date = urlParams.get('start_date');
        }
        if (urlParams.has('end_date') && !params.end_date) {
            params.end_date = urlParams.get('end_date');
        }
        if (urlParams.has('adults') && !params.adults) {
            params.adults = parseInt(urlParams.get('adults')) || 1;
        }
        if (urlParams.has('children') && !params.children) {
            params.children = parseInt(urlParams.get('children')) || 0;
        }
        if (urlParams.has('child_ages[]') || urlParams.has('children_ages[]')) {
            const ages = urlParams.getAll('child_ages[]').concat(urlParams.getAll('children_ages[]'));
            if (ages.length > 0) {
                params.child_ages = ages;
                params.children_ages = ages;
            }
        }

        // Try to get from search form if it exists (use form values as primary source if available)
        const $searchForm = $('#booking-hotels-room-search-form, form.room-search-form');
        if ($searchForm.length) {
            // Get values from form inputs (including hidden inputs)
            const startDate = $searchForm.find('[name="start_date"]').val();
            const endDate = $searchForm.find('[name="end_date"]').val();
            const adults = $searchForm.find('[name="adults"]').val();
            const children = $searchForm.find('[name="children"]').val();
            const childAgesInputs = $searchForm.find('[name="child_ages[]"]').map(function() {
                return $(this).val();
            }).get().filter(age => age !== '' && age !== null);

            if (startDate) params.start_date = startDate;
            if (endDate) params.end_date = endDate;
            if (adults) params.adults = parseInt(adults) || 1;
            if (children !== undefined && children !== null && children !== '') {
                params.children = parseInt(children) || 0;
            }
            if (childAgesInputs.length > 0) {
                params.child_ages = childAgesInputs;
                params.children_ages = childAgesInputs;
            }
        }

        // Update instance variable
        this.searchParams = params;

        return params;
    }

    /**
     * Clear all selections
     */
    handleClearSelection(e) {
        e.preventDefault();
        if (confirm(this.translate('Are you sure you want to clear all selections?'))) {
            this.selectedRooms = [];

        // Reset all plan selections - support both classes
        $('.rate-plan-item, .rate-plan-card-improved').removeClass('selected');
        $('.select-plan-btn')
            .removeClass('btn-success')
            .addClass('btn-primary')
            .html('<i class="fa fa-check"></i> ' + this.translate('Select'))
            .css('display', ''); // Remove inline styles, let CSS handle it
        // Remove inline styles from Book Now buttons - CSS rule will handle it when .selected is removed
        $('.book-now-btn').css('display', '');

        // Reset quantities to 1 (support both old and new selectors)
        $('.room-quantity-select, .room-quantity').val(1);

            this.updateSummary();
        }
    }

    /**
     * Update summary display
     */
    updateSummary() {
        const $summary = $('#rooms-selection-summary');
        const $roomsList = $('#selected-rooms-list');
        const $totalPrice = $('#total-booking-price');
        const $proceedBtn = $('#proceed-to-checkout-btn');

        if (this.selectedRooms.length === 0) {
            $summary.hide();
            return;
        }

        $summary.show();

        // Calculate total price
        const totalPrice = this.selectedRooms.reduce((sum, room) => sum + room.totalPrice, 0);
        $totalPrice.text(this.formatMoney(totalPrice));

        // Update rooms list
        let html = '';
        this.selectedRooms.forEach((room, index) => {
            const nights = this.searchParams.nights || 1;
            html += `
                <div class="selected-room-item" data-room-id="${room.roomId}">
                    <div class="room-info">
                        <div class="room-name">${room.roomName}</div>
                        <div class="plan-name">${room.planName}</div>
                        <div class="room-details">
                            <span><i class="fa fa-hashtag"></i> ${room.quantity} ${room.quantity == 1 ? this.translate('Room') : this.translate('Rooms')}</span>
                            ${room.mealOption ? `<span><i class="fa fa-cutlery"></i> ${room.mealOption.name}</span>` : ''}
                            ${room.isRefundable ? `<span class="text-success"><i class="fa fa-shield"></i> ${this.translate('Refundable')}</span>` : `<span class="text-warning"><i class="fa fa-exclamation-triangle"></i> ${this.translate('Non-refundable')}</span>`}
                        </div>
                    </div>
                    <div class="room-price">
                        <div class="price-amount">${this.formatMoney(room.totalPrice)}</div>
                        <div class="price-details">
                            ${this.formatMoney(room.pricePerNight)} / ${this.translate('night')}
                        </div>
                    </div>
                    <button type="button" class="remove-room" data-room-id="${room.roomId}">
                        <i class="fa fa-times"></i>
                    </button>
                </div>
            `;
        });

        $roomsList.html(html);

        // Enable/disable proceed button
        $proceedBtn.prop('disabled', this.selectedRooms.length === 0);
    }

    /**
     * Handle proceed to checkout
     */
    handleProceedToCheckout(e) {
        e.preventDefault();

        if (this.selectedRooms.length === 0) {
            alert(this.translate('Please select at least one room'));
            return;
        }

        // Build checkout URL with selected rooms
        const checkoutUrl = this.buildCheckoutUrl();
        if (checkoutUrl) {
            console.log('Navigating to checkout:', checkoutUrl); // ✅ Debug log
            window.location.href = checkoutUrl;
        } else {
            console.error('Failed to build checkout URL');
            alert(this.translate('Failed to build checkout URL. Please try again.'));
        }
    }

    /**
     * Build checkout URL with selected rooms
     */
    buildCheckoutUrl() {
        const baseUrl = window.bookingCheckoutUrl || '';
        if (!baseUrl) {
            console.error('Checkout URL not defined. window.bookingCheckoutUrl is:', window.bookingCheckoutUrl);
            console.error('Available window variables:', Object.keys(window).filter(k => k.includes('booking')));
            return null;
        }

        console.log('Building checkout URL with baseUrl:', baseUrl); // ✅ Debug log
        console.log('Current searchParams:', this.searchParams); // ✅ Debug log

        const params = new URLSearchParams();

        // ✅ FIX: Add all search params using centralized method
        this.addSearchParamsToUrl(params);

        // Add selected rooms
        // Format: rooms[0][room_id]=1&rooms[0][rate_plan_id]=1&rooms[0][quantity]=2
        console.log('✅ Building checkout URL with selected rooms:', this.selectedRooms); // ✅ Debug log

        if (this.selectedRooms.length === 0) {
            console.error('No rooms selected');
            return null;
        }

        this.selectedRooms.forEach((room, index) => {
            if (!room.roomId || !room.ratePlanId) {
                console.error('Invalid room data at index', index, room);
                this.logToTestFile('ERROR', 'Invalid room data', { index, room });
                return;
            }

            const roomId = room.roomId;
            const ratePlanId = room.ratePlanId;
            const quantity = room.quantity || 1;

            params.append(`rooms[${index}][room_id]`, roomId);
            params.append(`rooms[${index}][rate_plan_id]`, ratePlanId);
            params.append(`rooms[${index}][quantity]`, quantity);

            if (room.offerToken) {
                params.append(`rooms[${index}][offer_token]`, room.offerToken);
            }

            console.log(`✅ Added room ${index} to URL params:`, { // ✅ Debug log
                room_id: roomId,
                rate_plan_id: ratePlanId,
                quantity: quantity
            });
        });

        const finalUrl = baseUrl + '?' + params.toString();
        console.log('✅ Final checkout URL:', finalUrl); // ✅ Debug log

        // ✅ TEST LOG: Checkout URL built for multiple rooms
        this.logToTestFile('INFO', 'Checkout URL built (Multiple Rooms)', {
            selectedRooms: this.selectedRooms,
            searchParams: this.searchParams,
            finalUrl: finalUrl,
            urlParams: Object.fromEntries(params),
            totalRooms: this.selectedRooms.length,
            totalPrice: this.selectedRooms.reduce((sum, r) => sum + r.totalPrice, 0)
        });

        return finalUrl;
    }

    /**
     * Get booking data
     */
    getBookingData() {
        return {
            rooms: this.selectedRooms,
            totalPrice: this.selectedRooms.reduce((sum, r) => sum + r.totalPrice, 0),
            searchParams: this.searchParams,
        };
    }

    /**
     * Format money
     */
    formatMoney(amount) {
        if (typeof window.formatMoney === 'function') {
            return window.formatMoney(amount);
        }
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: window.bookingCurrency || 'USD'
        }).format(amount);
    }

    /**
     * Translate string
     */
    translate(key) {
        // Simple translation - replace with actual translation function if available
        const translations = {
            'Select': 'Select',
            'Selected': 'Selected',
            'Room': 'Room',
            'Rooms': 'Rooms',
            'night': 'night',
            'Refundable': 'Refundable',
            'Non-refundable': 'Non-refundable',
            'Are you sure you want to clear all selections?': 'Are you sure you want to clear all selections?',
            'Please select at least one room': 'Please select at least one room',
        };
        return translations[key] || key;
    }
}

// Initialize when document is ready
$(document).ready(function() {
    if (typeof window.roomSelectionManager === 'undefined') {
        window.roomSelectionManager = new RoomSelectionManager();
    }
});
